<?php namespace App\Controllers\Api\Customer\Payment\Paystack;

use App\Controllers\PrivateController;
use App\Libraries\Settings;
use App\Models\AppsModel;
use App\Models\DepositMethodsModel;
use App\Models\PlansExtraModel;
use App\Models\SubscribesExtraModel;
use App\Models\SubscribesModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class CreatePaymentExtraRequest extends PrivateController
{
    protected $paystackUrl = 'https://api.paystack.co/transaction/initialize';
    protected $method;

    public function __construct()
    {
        // Không gọi parent::__construct() nếu lớp cha không có phương thức khởi tạo
        $depositMethods = new DepositMethodsModel();
        $this->method = $depositMethods->where([
            'name' => 'Paystack',
            'status' => 1
        ])->first();

        if (!$this->method) {
            throw new DatabaseException('Paystack method not found or inactive.');
        }
    }

    public function index(): ResponseInterface
    {
        // Kiểm tra dữ liệu đầu vào
        if (!$this->validate($this->create_validation_type())) {
            return $this->respond(["message" => $this->validator->getErrors()], 400);
        }

        $subscribe_uid = esc($this->request->getVar("uid"));
        log_message('debug', 'Received UID: ' . $subscribe_uid);
        $plan_extra_id = esc($this->request->getVar("plan_extra_id"));
        log_message('debug', 'Received plan_extra_id: ' . $plan_extra_id);

        $db = \Config\Database::connect();
        $db->transStart();

        try {
            $app = $this->getAppFromSubscribe($subscribe_uid);
            // Đã loại bỏ kiểm tra đăng ký hoạt động
            $extraPlan = $this->getPlanExtra($plan_extra_id);
            $payment = $this->createPaystackPaymentExtra($extraPlan, $app["id"]);

            if (isset($payment['data']['authorization_url'])) {
                $db->transComplete();
                return $this->respond(["url" => $payment['data']['authorization_url']], 200);
            } else {
                throw new DatabaseException("Paystack payment creation failed.");
            }
        } catch (\Exception $e) {
            $db->transRollback();
            log_message('error', 'CreatePaymentExtraRequest Error: ' . $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function getAppFromSubscribe($subscribe_uid)
    {
        $subscribes = new SubscribesModel();
        $subscribe = $subscribes
            ->where('uid', $subscribe_uid)
            ->select("app_id")
            ->first();

        if (!$subscribe) {
            log_message('error', "Subscribe not found with UID: $subscribe_uid");
            throw new DatabaseException("Subscribe not found with UID: $subscribe_uid");
        }

        $projects = new AppsModel();
        $app = $projects->where([
            "id" => $subscribe['app_id'],
            "user" => $this->userId,
            "deleted_at" => 0
        ])->select("id")->first();

        if (!$app) {
            log_message('error', "App not found for app_id: " . $subscribe['app_id']);
            throw new DatabaseException("App not found!");
        }

        return $app;
    }

    private function getPlanExtra($plan_extra_id)
    {
        $plansExtra = new PlansExtraModel();
        $extraPlan = $plansExtra->where([
            "id" => $plan_extra_id,
            "status" => 1,
            "deleted_at" => 0
        ])->first();

        if (!$extraPlan) {
            throw new DatabaseException(lang("Message.message_83"));
        }

        return $extraPlan;
    }

    private function createPaystackPaymentExtra($extraPlan, $appId)
    {
        $settings = new Settings();
        $frontUrl = rtrim($settings->get_config("site_url"), '/') . '/';

        // Đặt callback_url thành URL bạn muốn người dùng được chuyển hướng sau khi thanh toán thành công
        $callbackUrl = 'https://app.appfast.net/private/profile/subscribe';

        $secretKey = $this->method["api_value_2"]; // Paystack Secret Key

        // Lấy mã tiền tệ từ bảng settings (nếu cần thiết)
        $currencyCode = $settings->get_config("currency_code");

        log_message('debug', 'Paystack Secret Key: ' . $secretKey);
        log_message('debug', 'Currency Code: ' . $currencyCode);

        $client = \Config\Services::curlrequest([
            'base_uri' => $this->paystackUrl,
            'headers' => [
                'Authorization' => "Bearer $secretKey",
                'Content-Type' => 'application/json'
            ]
        ]);

        // Tạo custom_id với tiền tố 'EXTRA_'
        $metadata = [
            'custom_id' => 'EXTRA_' . $extraPlan["id"] . "_" .  $this->userId . "_" . $appId
        ];

        $response = $client->post('', [
            'json' => [
                'amount' => $extraPlan["price"] * 100, // Paystack yêu cầu số tiền là integer (kobo)
                'email' => $this->getUserEmail(),
                'callback_url' => $callbackUrl,
                'metadata' => $metadata
            ]
        ]);

        $paymentResponse = json_decode($response->getBody(), true);
        log_message('debug', 'Paystack Payment Response: ' . json_encode($paymentResponse));

        if (!isset($paymentResponse['status']) || !$paymentResponse['status']) {
            throw new DatabaseException("Paystack payment initialization failed.");
        }

        return $paymentResponse;
    }

    private function getUserEmail()
    {
        $users = new \App\Models\UsersModel();
        $user = $users->where('id', $this->userId)->select('email')->first();
        if (!$user) {
            throw new \Exception("User not found.");
        }
        return $user['email'];
    }

    private function create_validation_type(): array
    {
        return [
            "plan_extra_id" => [
                "label" => lang("Fields.field_149"), // Bạn có thể định nghĩa ngôn ngữ phù hợp
                "rules" => "required|numeric"
            ],
            "uid" => [
                "label" => "UID", // Bạn có thể định nghĩa ngôn ngữ phù hợp
                "rules" => "required|string"
            ],
        ];
    }
}
